<?php

// Функция для проверки, содержит ли сообщение ошибку в формате 7xxxxxxxxxx пояснительный текст
function extractErrors($message)
{
    $pattern = '/^(7\d{10}|375\d{9})\s+.+$/m'; // Регулярное выражение для поиска всех строк с форматом ошибки
    preg_match_all($pattern, trim($message), $matches);
    file_put_contents('log.txt', "Extracted errors: " . print_r($matches[0], true) . "\n", FILE_APPEND);
    return $matches[0]; // Возвращаем массив ошибок
}

// Функция для фиксации ошибки и отправки в чаты менеджеров
function handleClientMessage($pdo, $chatId, $message)
{
    // Проверяем, существует ли клиент
    $clientExists = checkIfClientExists($pdo, $chatId);

    // Если клиент не существует, добавляем его в базу данных
    if (!$clientExists) {
        addClientToDatabase($pdo, $chatId, $message['from']['first_name']);
    }

    // Извлекаем ошибки из сообщения
    $errors = extractErrors($message['text']);
    if (!empty($errors)) {
        sendReaction($chatId, $message['message_id'], '✍️');
        foreach ($errors as $error) {
            $errorId = addErrorToDatabase($pdo, $chatId, ['text' => $error, 'message_id' => $message['message_id']]);
            if ($errorId) {
                $managerChatIds = getManagerChatIds($pdo);
                if (!empty($managerChatIds)) {
                    foreach ($managerChatIds as $managerChatId) {
                        if ($managerChatId != 0) {
                            sendErrorToManager($managerChatId, $error, $errorId); // Используем правильный id
                        }
                    }
                    file_put_contents('log.txt', "Error sent to managers.\n", FILE_APPEND);
                } else {
                    file_put_contents('log.txt', "No managers found to send the error.\n", FILE_APPEND);
                }
            } else {
                file_put_contents('log.txt', "Failed to add error to database.\n", FILE_APPEND);
            }
        }
    } else {
        file_put_contents('log.txt', "Сообщение не соответствует формату 7xxxxxxxxxx пояснительный текст.\n", FILE_APPEND);
    }
}


// Функция для добавления реакции на сообщение
function sendReaction($chatId, $messageId, $emoji)
{
    global $config;
    $token = $config['telegram']['token'];
    $url = "https://api.telegram.org/bot$token/setMessageReaction";

    // Подготовка данных для запроса
    $reactionData = [
        [
            'type' => 'emoji',
            'emoji' => $emoji
        ]
    ];

    $postFields = [
        'chat_id' => $chatId,
        'message_id' => $messageId,
        'reaction' => json_encode($reactionData)
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_HTTPHEADER, array("Content-Type:application/json"));
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($postFields));
    $output = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    // Логируем результат
    if ($httpCode == 200 && json_decode($output)->ok) {
        file_put_contents('log.txt', "Reaction $emoji added to chat $chatId, message $messageId\n", FILE_APPEND);
    } else {
        file_put_contents('log.txt', "Failed to add reaction $emoji to chat $chatId, message $messageId. HTTP Code: $httpCode. Response: $output\n", FILE_APPEND);
    }
}

function addErrorToDatabase($pdo, $chatId, $message)
{
    try {
        $stmt = $pdo->prepare('SELECT id FROM clients WHERE chat_id = :chat_id LIMIT 1');
        $stmt->execute(['chat_id' => $chatId]);
        $client = $stmt->fetch();

        if (!$client) {
            file_put_contents('log.txt', "Client with chat_id $chatId not found.\n", FILE_APPEND);
            return false;
        }

        $stmt = $pdo->prepare('INSERT INTO errors (client_id, description, client_message_id, manager_id) VALUES (:client_id, :description, :client_message_id, :manager_id)');
        $stmt->execute([
            'client_id' => $client['id'],
            'description' => $message['text'],
            'client_message_id' => $message['message_id'],
            'manager_id' => '' // или NULL, если поле допускает NULL
        ]);

        // Получаем последний вставленный ID
        $errorId = $pdo->lastInsertId();
        file_put_contents('log.txt', "Error with id $errorId added to database for client with chat_id $chatId.\n", FILE_APPEND);

        return $errorId;
    } catch (\PDOException $e) {
        file_put_contents('log.txt', 'Failed to add error to database: ' . $e->getMessage() . "\n", FILE_APPEND);
        return false;
    }
}


function sendErrorToManager($managerChatId, $errorMessage, $errorId)
{
    global $config;
    $token = $config['telegram']['token'];
    $url = "https://api.telegram.org/bot$token/sendMessage";

    $pattern = '/^(7\d{10}|375\d{9})\s+(.+)$/';
    if (preg_match($pattern, $errorMessage, $matches)) {
        $phoneNumber = $matches[1];
        $errorText = $matches[2];
    } else {
        $phoneNumber = 'Не найден';
        $errorText = $errorMessage;
    }

    $replyMarkup = [
        'inline_keyboard' => [
            [
                [
                    'text' => 'Обрабатываю',
                    'callback_data' => "process_error_$errorId"
                ],
                [
                    'text' => 'Не найдено',
                    'callback_data' => "not_found_error_$errorId" // Новый callback_data
                ],
                [
                    'text' => 'Исправлено',
                    'callback_data' => "fix_error_$errorId"
                ]
            ]
        ]
    ];


    $messageText = "<b>Номер:</b> <code>$phoneNumber</code>\n<b>Текст:</b> <code>$errorText</code>";
    $postFields = [
        'chat_id' => $managerChatId,
        'text' => $messageText,
        'reply_markup' => json_encode($replyMarkup),
        'parse_mode' => "html"
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_HTTPHEADER, array("Content-Type:application/json"));
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($postFields));
    $output = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    file_put_contents('log.txt', "Sent error message to manager chat $managerChatId with HTTP code $httpCode. Response: $output. cURL Error: $curlError\n", FILE_APPEND);
}

function sendntf($managerChatId, $errorMessage, $reply_to_message_id)
{
    global $config;
    $token = $config['telegram']['token'];
    $url = "https://api.telegram.org/bot$token/sendMessage";

    $pattern = '/^(7\d{10}|375\d{9})\s+(.+)$/';
    if (preg_match($pattern, $errorMessage, $matches)) {
        $phoneNumber = $matches[1];
        $errorText = $matches[2];
    } else {
        $phoneNumber = 'Не найден';
        $errorText = $errorMessage;
    }


    $messageText = "<b>Поставщик исправил ошибку</b>\n<b>Номер:</b> <code>$phoneNumber</code>\n<b>Текст:</b> <code>$errorText</code>";
    $postFields = [
        'chat_id' => $managerChatId,
        'text' => $messageText,
        'parse_mode' => "html",
        'reply_to_message_id' => $reply_to_message_id
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_HTTPHEADER, array("Content-Type:application/json"));
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($postFields));
    $output = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    file_put_contents('log.txt', "Sent error message to manager chat $managerChatId with HTTP code $httpCode. Response: $output. cURL Error: $curlError\n", FILE_APPEND);
}

// Функция для отправки ошибки поставщику
function sendErrorToSupplier($supplierChatId, $errorMessage, $originalMessageId)
{
    global $config;
    $token = $config['telegram']['token'];
    $url = "https://api.telegram.org/bot$token/sendMessage";

    // Создание inline-кнопок
    $replyMarkup = [
        'inline_keyboard' => [
            [
                [
                    'text' => 'Исправлено',
                    'callback_data' => "fix_supplier_error_$originalMessageId"
                ]
            ]
        ]
    ];
    $pattern = '/^(\S+)\s+(.+)$/';
    if (preg_match($pattern, $errorMessage, $matches)) {
        $firstPart = $matches[1]; // Часть до первого пробела
        $secondPart = $matches[2]; // Часть после пробела
    } else {
        // Если формат не соответствует ожидаемому, отправляем сообщение целиком
        $firstPart = 'Не найдено';
        $secondPart = $errorMessage;
    }


    $messageText =  "<b>Номер:</b> <code>$firstPart</code>\n<b>Текст:</b> <code>$secondPart</code>";
    $postFields = [
        'chat_id' => $supplierChatId,
        'text' => $messageText,
        'reply_markup' => json_encode($replyMarkup),
        'parse_mode' => "html"
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_HTTPHEADER, array("Content-Type:application/json"));
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($postFields));
    $output = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);  // Получаем ошибки cURL, если они есть
    curl_close($ch);

    // Логируем HTTP код, ответ API и ошибки cURL
    file_put_contents('log.txt', "Sent error message to supplier chat $supplierChatId with HTTP code $httpCode. Response: $output. cURL Error: $curlError\n", FILE_APPEND);
}

// Функция для проверки, существует ли клиент
function checkIfClientExists($pdo, $chatId)
{
    try {
        $stmt = $pdo->prepare('SELECT id FROM clients WHERE chat_id = :chat_id LIMIT 1');
        $stmt->execute(['chat_id' => $chatId]);
        $client = $stmt->fetch();

        return $client !== false;
    } catch (\PDOException $e) {
        file_put_contents('log.txt', 'Failed to check if client exists: ' . $e->getMessage() . "\n", FILE_APPEND);
        return false;
    }
}

// Функция для добавления клиента в базу данных
function addClientToDatabase($pdo, $chatId, $name)
{
    try {
        $stmt = $pdo->prepare('INSERT INTO clients (chat_id, name) VALUES (:chat_id, :name)');
        $stmt->execute(['chat_id' => $chatId, 'name' => $name]);

        file_put_contents('log.txt', "Client with chat_id $chatId and name $name added to database.\n", FILE_APPEND);
    } catch (\PDOException $e) {
        file_put_contents('log.txt', 'Failed to add client to database: ' . $e->getMessage() . "\n", FILE_APPEND);
    }
}

// Функция для получения chat_id всех менеджеров
function getManagerChatIds($pdo)
{
    try {
        $stmt = $pdo->query('SELECT chat_id FROM managers');
        $managerChatIds = $stmt->fetchAll(PDO::FETCH_COLUMN);
        file_put_contents('log.txt', "Fetched manager chat IDs: " . implode(', ', $managerChatIds) . "\n", FILE_APPEND);
        return $managerChatIds;
    } catch (\PDOException $e) {
        file_put_contents('log.txt', 'Failed to fetch manager chat IDs: ' . $e->getMessage() . "\n", FILE_APPEND);
        return [];
    }
}

// Функция для получения chat_id поставщиков
function getSupplierChatIds($pdo)
{
    try {
        $stmt = $pdo->query('SELECT chat_id FROM suppliers');
        $supplierChatIds = $stmt->fetchAll(PDO::FETCH_COLUMN);
        file_put_contents('log.txt', "Fetched supplier chat IDs: " . implode(', ', $supplierChatIds) . "\n", FILE_APPEND);
        return $supplierChatIds;
    } catch (\PDOException $e) {
        file_put_contents('log.txt', 'Failed to fetch supplier chat IDs: ' . $e->getMessage() . "\n", FILE_APPEND);
        return [];
    }
}
function isManager($pdo, $chatId)
{
    try {
        $stmt = $pdo->prepare('SELECT id FROM managers WHERE chat_id = :chat_id LIMIT 1');
        $stmt->execute(['chat_id' => $chatId]);
        return $stmt->fetch() !== false;
    } catch (\PDOException $e) {
        file_put_contents('log.txt', 'Failed to check if chat_id is manager: ' . $e->getMessage() . "\n", FILE_APPEND);
        return false;
    }
}

// Функция для проверки, является ли chat_id поставщиком
function isSupplier($pdo, $chatId)
{
    try {
        $stmt = $pdo->prepare('SELECT id FROM suppliers WHERE chat_id = :chat_id LIMIT 1');
        $stmt->execute(['chat_id' => $chatId]);
        return $stmt->fetch() !== false;
    } catch (\PDOException $e) {
        file_put_contents('log.txt', 'Failed to check if chat_id is supplier: ' . $e->getMessage() . "\n", FILE_APPEND);
        return false;
    }
}
