<?php
// test.php - Тест производительности сервера
header('Content-Type: text/plain; charset=utf-8');

echo "=== ТЕСТ ПРОИЗВОДИТЕЛЬНОСТИ СЕРВЕРА ===\n\n";

// 1. Информация о сервере
echo "1. ИНФОРМАЦИЯ О СЕРВЕРЕ:\n";
echo "   PHP Version: " . PHP_VERSION . "\n";
echo "   OS: " . PHP_OS . "\n";
echo "   Memory Limit: " . ini_get('memory_limit') . "\n";
echo "   Max Execution Time: " . ini_get('max_execution_time') . "s\n";
echo "   Server Software: " . ($_SERVER['SERVER_SOFTWARE'] ?? 'N/A') . "\n\n";

// 2. Тест скорости процессора
echo "2. ТЕСТ СКОРОСТИ ПРОЦЕССОРА:\n";
$start = microtime(true);
$operations = 0;
for ($i = 0; $i < 1000000; $i++) {
    $operations += sqrt($i) * sin($i) / cos($i + 1);
}
$cpu_time = microtime(true) - $start;
echo "   1,000,000 математических операций: " . number_format($cpu_time, 4) . " сек\n";
echo "   Операций в секунду: " . number_format(1000000 / $cpu_time) . "\n\n";

// 3. Тест оперативной памяти
echo "3. ТЕСТ ОПЕРАТИВНОЙ ПАМЯТИ:\n";
$start = microtime(true);
$bigArray = [];
for ($i = 0; $i < 100000; $i++) {
    $bigArray[] = str_repeat('test', 10) . $i;
}
$memory_usage = memory_get_usage(true);
$memory_time = microtime(true) - $start;
echo "   Создание массива из 100,000 элементов: " . number_format($memory_time, 4) . " сек\n";
echo "   Использовано памяти: " . number_format($memory_usage / 1024 / 1024, 2) . " MB\n\n";
unset($bigArray);

// 4. Тест файловой системы
echo "4. ТЕСТ ФАЙЛОВОЙ СИСТЕМЫ:\n";
$start = microtime(true);
$testFile = 'test_speed.txt';
$content = str_repeat("Тест скорости записи файлов\n", 1000);
file_put_contents($testFile, $content);
$write_time = microtime(true) - $start;

$start = microtime(true);
$readContent = file_get_contents($testFile);
$read_time = microtime(true) - $start;

unlink($testFile);

echo "   Запись файла (30KB): " . number_format($write_time, 4) . " сек\n";
echo "   Чтение файла (30KB): " . number_format($read_time, 4) . " сек\n\n";

// 5. Тест базы данных (если доступна)
echo "5. ТЕСТ БАЗЫ ДАННЫХ:\n";
try {
    $config = require 'config.php';
    $dsn = "mysql:host={$config['database']['host']};dbname={$config['database']['db']};charset={$config['database']['charset']}";
    $pdo = new PDO($dsn, $config['database']['user'], $config['database']['pass'], [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_TIMEOUT => 5
    ]);
    
    $start = microtime(true);
    $stmt = $pdo->query("SELECT 1 as test");
    $result = $stmt->fetch();
    $db_time = microtime(true) - $start;
    
    echo "   Подключение к БД и простой запрос: " . number_format($db_time, 4) . " сек\n";
    echo "   Статус БД: ✓ Доступна\n\n";
} catch (Exception $e) {
    echo "   Статус БД: ✗ Не доступна (" . $e->getMessage() . ")\n\n";
}

// 6. Тест cURL (запрос к самому себе)
echo "6. ТЕСТ HTTP ЗАПРОСОВ:\n";
$start = microtime(true);
$ch = curl_init();
curl_setopt_array($ch, [
    CURLOPT_URL => 'http://' . $_SERVER['HTTP_HOST'] . $_SERVER['SCRIPT_NAME'],
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => 10,
    CURLOPT_FOLLOWLOCATION => true
]);
$response = curl_exec($ch);
$http_time = microtime(true) - $start;
$http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

echo "   HTTP запрос к серверу: " . number_format($http_time, 4) . " сек\n";
echo "   HTTP код: " . $http_code . "\n\n";

// 7. Тест запросов к внешним API
echo "7. ТЕСТ ЗАПРОСОВ К ВНЕШНИМ СЕРВИСАМ:\n";
$test_urls = [
    'Google' => 'https://www.google.com',
    'Telegram API' => 'https://api.telegram.org',
    'Localhost' => 'http://localhost'
];

foreach ($test_urls as $name => $url) {
    $start = microtime(true);
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 5,
        CURLOPT_NOBODY => true, // Только HEAD запрос
        CURLOPT_FOLLOWLOCATION => true
    ]);
    curl_exec($ch);
    $time = microtime(true) - $start;
    curl_close($ch);
    
    echo "   $name: " . number_format($time, 3) . " сек\n";
}
echo "\n";

// 8. Итоговая оценка
echo "8. ИТОГОВАЯ ОЦЕНКА ПРОИЗВОДИТЕЛЬНОСТИ:\n";

$scores = [];
if ($cpu_time < 0.1) $scores[] = "✓ Процессор: Отлично";
elseif ($cpu_time < 0.3) $scores[] = "✓ Процессор: Хорошо"; 
else $scores[] = "⚠ Процессор: Медленно";

if ($memory_time < 0.05) $scores[] = "✓ Память: Отлично";
elseif ($memory_time < 0.1) $scores[] = "✓ Память: Хорошо";
else $scores[] = "⚠ Память: Медленно";

if ($write_time < 0.01) $scores[] = "✓ Файлы: Отлично";
elseif ($write_time < 0.05) $scores[] = "✓ Файлы: Хорошо";
else $scores[] = "⚠ Файлы: Медленно";

if (isset($db_time) && $db_time < 0.1) $scores[] = "✓ БД: Отлично";
elseif (isset($db_time) && $db_time < 0.3) $scores[] = "✓ БД: Хорошо";
else $scores[] = "⚠ БД: Медленно/Недоступна";

if ($http_time < 0.5) $scores[] = "✓ HTTP: Отлично";
elseif ($http_time < 1.0) $scores[] = "✓ HTTP: Хорошо";
else $scores[] = "⚠ HTTP: Медленно";

foreach ($scores as $score) {
    echo "   " . $score . "\n";
}

echo "\n=== ТЕСТ ЗАВЕРШЕН ===\n";

// 9. Рекомендации
echo "\nРЕКОМЕНДАЦИИ:\n";
if ($cpu_time > 0.3) echo "- Рассмотреть более мощный CPU\n";
if ($memory_time > 0.1) echo "- Увеличить memory_limit в php.ini\n"; 
if ($write_time > 0.05) echo "- Проверить скорость диска\n";
if ($http_time > 1.0) echo "- Оптимизировать HTTP запросы\n";

echo "\nТекущее время: " . date('Y-m-d H:i:s') . "\n";
?>