<?php
header('Content-Type: application/json');

/**
 * Подключение к базе данных
 */
function getPDOConnection($config) {
    $dsn = "mysql:host={$config['database']['host']};dbname={$config['database']['db']};charset={$config['database']['charset']}";
    $options = [
        PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES   => false,
    ];

    try {
        $pdo = new PDO($dsn, $config['database']['user'], $config['database']['pass'], $options);
        return $pdo;
    } catch (\PDOException $e) {
        error_log('Database connection failed: ' . $e->getMessage());
        throw $e;
    }
}

/**
 * Функция отправки уведомления пользователю через Telegram Bot
 */
function sendTelegramNotification($userId, $message = null) {
    try {
        $config = require 'config.php';
        $botToken = $config['telegram']['token'];
        
        if (empty($botToken)) {
            error_log("Telegram bot token not found in config");
            return false;
        }
        
        $text = "📨 *Вам пришло сообщение от поддержки*";
        
        if ($message && strlen($message) > 0) {
            // Обрезаем сообщение если слишком длинное и экранируем markdown
            $preview = strlen($message) > 50 ? substr($message, 0, 50) . "..." : $message;
            $preview = str_replace(['_', '*', '[', ']', '(', ')', '~', '`', '>', '#', '+', '-', '=', '|', '{', '}', '.', '!'], 
                                 ['\\_', '\\*', '\\[', '\\]', '\\(', '\\)', '\\~', '\\`', '\\>', '\\#', '\\+', '\\-', '\\=', '\\|', '\\{', '\\}', '\\.', '\\!'], 
                                 $preview);
            $text .= "\n\n💬 " . $preview;
        }
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    [
                        'text' => '💬 Открыть чат',
                        'url' => 'https://t.me/LegenDvoipSupportBot/support'
                    ]
                ]
            ]
        ];
        
        $data = [
            'chat_id' => $userId,
            'text' => $text,
            'parse_mode' => 'Markdown',
            'reply_markup' => json_encode($keyboard)
        ];
        
        $url = "https://api.telegram.org/bot{$botToken}/sendMessage";
        
        $options = [
            'http' => [
                'header' => "Content-type: application/x-www-form-urlencoded\r\n",
                'method' => 'POST',
                'content' => http_build_query($data),
                'timeout' => 10
            ]
        ];
        
        $context = stream_context_create($options);
        $result = @file_get_contents($url, false, $context);
        
        if ($result === false) {
            error_log("Failed to send Telegram notification to user {$userId}");
            return false;
        }
        
        $response = json_decode($result, true);
        
        if (!$response['ok']) {
            error_log("Telegram API error for user {$userId}: " . $response['description']);
            return false;
        }
        
        error_log("Telegram notification sent successfully to user {$userId}");
        return true;
        
    } catch (Exception $e) {
        error_log("Exception sending Telegram notification: " . $e->getMessage());
        return false;
    }
}

/**
 * Функция отправки уведомления менеджеру о новом сообщении от клиента
 */
function sendManagerNotification($clientId, $message = null) {
    try {
        $config = require 'config.php';
        $botToken = $config['telegram']['token'];
        
        if (empty($botToken)) {
            error_log("ERROR: Bot token is empty");
            return false;
        }
        
        // Получаем ID менеджеров из базы данных
        $pdo = getPDOConnection($config);
        $stmt = $pdo->prepare('SELECT chat_id FROM managers');
        $stmt->execute();
        $managers = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        // Добавляем отладочную информацию
        error_log("DEBUG: Bot token exists: yes");
        error_log("DEBUG: Manager IDs from database: " . print_r($managers, true));
        error_log("DEBUG: Client ID: " . $clientId);
        error_log("DEBUG: Message: " . $message);
        
        if (empty($managers)) {
            error_log("ERROR: No managers found in database");
            return false;
        }
        
        $clientIdShort = substr($clientId, -4);
        $text = "🆘 *Новое сообщение от клиента {$clientIdShort}*";
        
        if ($message && strlen($message) > 0) {
            // Обрезаем сообщение если слишком длинное и экранируем markdown
            $preview = strlen($message) > 60 ? substr($message, 0, 60) . "..." : $message;
            $preview = str_replace(['_', '*', '[', ']', '(', ')', '~', '`', '>', '#', '+', '-', '=', '|', '{', '}', '.', '!'], 
                                 ['\\_', '\\*', '\\[', '\\]', '\\(', '\\)', '\\~', '\\`', '\\>', '\\#', '\\+', '\\-', '\\=', '\\|', '\\{', '\\}', '\\.', '\\!'], 
                                 $preview);
            $text .= "\n\n💬 " . $preview;
        }
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    [
                        'text' => '👨‍💼 Панель менеджера',
                        'url' => 'https://t.me/LegenDvoipSupportBot/addman'
                    ]
                ]
            ]
        ];
        
        $url = "https://api.telegram.org/bot{$botToken}/sendMessage";
        
        $success = false;
        
        // Отправляем уведомление всем менеджерам
        foreach ($managers as $managerId) {
            error_log("DEBUG: Sending notification to manager ID: " . $managerId);
            
            $data = [
                'chat_id' => $managerId,
                'text' => $text,
                'parse_mode' => 'Markdown',
                'reply_markup' => json_encode($keyboard)
            ];
            
            $options = [
                'http' => [
                    'header' => "Content-type: application/x-www-form-urlencoded\r\n",
                    'method' => 'POST',
                    'content' => http_build_query($data),
                    'timeout' => 10
                ]
            ];
            
            $context = stream_context_create($options);
            $result = @file_get_contents($url, false, $context);
            
            if ($result !== false) {
                $response = json_decode($result, true);
                error_log("DEBUG: Telegram API response for manager {$managerId}: " . $result);
                
                if ($response && $response['ok']) {
                    error_log("SUCCESS: Manager notification sent to {$managerId}");
                    $success = true;
                } else {
                    $errorDesc = $response['description'] ?? 'Unknown error';
                    error_log("ERROR: Telegram API error for manager {$managerId}: " . $errorDesc);
                }
            } else {
                error_log("ERROR: Failed to send HTTP request to manager {$managerId}");
            }
        }
        
        return $success;
        
    } catch (Exception $e) {
        error_log("EXCEPTION: " . $e->getMessage());
        return false;
    }
}

// Получение данных из POST
$userId = isset($_POST['user_id']) ? $_POST['user_id'] : null;
$text = isset($_POST['text']) ? $_POST['text'] : null;
$sender = isset($_POST['sender']) && in_array($_POST['sender'], ['user', 'support']) ? $_POST['sender'] : null;

// Отладочная информация
error_log("DEBUG: Received data - User ID: {$userId}, Sender: {$sender}, Text: " . substr($text, 0, 50));

if ($userId === null || $text === null || $sender === null) {
    error_log("ERROR: Missing required parameters");
    echo json_encode(['success' => false, 'error' => 'Missing parameters']);
    exit;
}

// Создаем папку support если не существует
if (!is_dir('support')) {
    mkdir('support', 0755, true);
}

$jsonPath = 'support/' . $userId . '.json';

// Создание файла, если он не существует
if (!file_exists($jsonPath)) {
    file_put_contents($jsonPath, json_encode(['messages' => []]), LOCK_EX);
    chmod($jsonPath, 0666);
}

// Чтение и запись с блокировкой
$file = fopen($jsonPath, 'c+');
if (!flock($file, LOCK_EX)) {
    fclose($file);
    error_log("ERROR: Failed to lock file {$jsonPath}");
    echo json_encode(['success' => false, 'error' => 'File lock failed']);
    exit;
}

$data = json_decode(fread($file, filesize($jsonPath) ?: 1), true);
if (!$data) {
    $data = ['messages' => []];
}

// Добавление нового сообщения
$newMessage = [
    'sender' => $sender,
    'text' => $text,
    'timestamp' => date('d.m.Y H:i:s')
];
$data['messages'][] = $newMessage;

// Запись обратно в файл
fseek($file, 0);
ftruncate($file, 0);
fwrite($file, json_encode($data, JSON_PRETTY_PRINT));
flock($file, LOCK_UN);
fclose($file);

error_log("SUCCESS: Message saved to file");

// 🔔 ОТПРАВЛЯЕМ УВЕДОМЛЕНИЯ
if ($sender === 'support') {
    // Менеджер ответил клиенту - уведомляем клиента
    error_log("DEBUG: Sending notification to client {$userId}");
    sendTelegramNotification($userId, $text);
} elseif ($sender === 'user') {
    // Клиент написал сообщение - уведомляем менеджеров
    error_log("DEBUG: Sending notification to managers about client {$userId}");
    sendManagerNotification($userId, $text);
}

echo json_encode(['success' => true]);
?>