<?php
// Подключение конфигурации для базы данных
$config = require 'config.php';

// Подключение к базе данных
function getPDOConnection($config)
{
    $dsn = "mysql:host={$config['database']['host']};dbname={$config['database']['db']};charset={$config['database']['charset']}";
    $options = [
        PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES   => false,
    ];

    try {
        $pdo = new PDO($dsn, $config['database']['user'], $config['database']['pass'], $options);
        file_put_contents('log.txt', "Database connection established.\n", FILE_APPEND);
        return $pdo;
    } catch (\PDOException $e) {
        file_put_contents('log.txt', 'Database connection failed: ' . $e->getMessage() . "\n", FILE_APPEND);
        die('Ошибка подключения к базе данных.');
    }
}

$pdo = getPDOConnection($config);

// Функция проверки, является ли пользователь менеджером
function isManager($pdo, $userId)
{
    try {
        $stmt = $pdo->prepare('SELECT id FROM managers WHERE chat_id = :chat_id LIMIT 1');
        $stmt->execute(['chat_id' => $userId]);
        return $stmt->fetch() !== false;
    } catch (\PDOException $e) {
        file_put_contents('log.txt', 'Failed to check if user_id is manager: ' . $e->getMessage() . "\n", FILE_APPEND);
        return false;
    }
}
?>

<!DOCTYPE html>
<html lang="ru" class="h-100">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Чат поддержки</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        * {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        }
        
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            height: 100vh;
            margin: 0;
            overflow: hidden;
        }
        
        .chat-container {
            height: 100vh;
            max-width: 500px;
            margin: 0 auto;
            display: flex;
            flex-direction: column;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            box-shadow: 0 25px 50px rgba(0, 0, 0, 0.15);
        }
        
        .chat-header {
            background: linear-gradient(135deg, #6c5ce7, #a29bfe);
            padding: 20px;
            border-radius: 0;
            position: relative;
            overflow: hidden;
        }
        
        .chat-header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grid" width="10" height="10" patternUnits="userSpaceOnUse"><path d="M 10 0 L 0 0 0 10" fill="none" stroke="rgba(255,255,255,0.1)" stroke-width="0.5"/></pattern></defs><rect width="100" height="100" fill="url(%23grid)"/></svg>');
            opacity: 0.3;
        }
        
        .chat-header h2 {
            margin: 0;
            color: white;
            font-weight: 600;
            font-size: 1.5rem;
            position: relative;
            z-index: 2;
            display: flex;
            align-items: center;
            gap: 12px;
        }
        
        .status-indicator {
            width: 12px;
            height: 12px;
            background: #00b894;
            border-radius: 50%;
            box-shadow: 0 0 10px rgba(0, 184, 148, 0.6);
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.1); }
            100% { transform: scale(1); }
        }
        
        .chat-messages {
            flex: 1;
            padding: 20px;
            overflow-y: auto;
            background: linear-gradient(to bottom, rgba(255, 255, 255, 0.1), rgba(247, 250, 255, 0.8));
            position: relative;
        }
        
        .chat-messages::-webkit-scrollbar {
            width: 6px;
        }
        
        .chat-messages::-webkit-scrollbar-track {
            background: rgba(0, 0, 0, 0.05);
            border-radius: 10px;
        }
        
        .chat-messages::-webkit-scrollbar-thumb {
            background: linear-gradient(135deg, #6c5ce7, #a29bfe);
            border-radius: 10px;
        }
        
        .message {
            max-width: 280px;
            margin-bottom: 16px;
            position: relative;
            opacity: 0;
            transform: translateY(10px);
            transition: all 0.3s ease-out;
        }
        
        .message.loaded {
            opacity: 1;
            transform: translateY(0);
        }
        
        .message.new {
            animation: slideIn 0.4s ease-out forwards;
        }
        
        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(20px) scale(0.95);
            }
            to {
                opacity: 1;
                transform: translateY(0) scale(1);
            }
        }
        
        .chat-messages.loading {
            opacity: 0.7;
            pointer-events: none;
        }
        
        .chat-messages.loaded {
            opacity: 1;
            transition: opacity 0.3s ease;
        }
        
        .message.user {
            margin-left: auto;
            margin-right: 0;
        }
        
        .message.support {
            margin-right: auto;
            margin-left: 0;
        }
        
        .message-bubble {
            padding: 14px 18px;
            border-radius: 20px;
            position: relative;
            word-wrap: break-word;
            box-shadow: 0 2px 12px rgba(0, 0, 0, 0.08);
        }
        
        .message.user .message-bubble {
            background: linear-gradient(135deg, #6c5ce7, #a29bfe);
            color: white;
            border-bottom-right-radius: 6px;
        }
        
        .message.user .message-bubble::after {
            content: '';
            position: absolute;
            bottom: 0;
            right: -8px;
            width: 0;
            height: 0;
            border: 8px solid transparent;
            border-left-color: #a29bfe;
            border-bottom: 0;
            transform: rotate(45deg);
        }
        
        .message.support .message-bubble {
            background: white;
            color: #2d3436;
            border-bottom-left-radius: 6px;
            border: 1px solid rgba(0, 0, 0, 0.06);
        }
        
        .message.support .message-bubble::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: -8px;
            width: 0;
            height: 0;
            border: 8px solid transparent;
            border-right-color: white;
            border-bottom: 0;
            transform: rotate(-45deg);
        }
        
        .timestamp {
            font-size: 0.75rem;
            opacity: 0.6;
            margin-top: 4px;
            text-align: center;
        }
        
        .message.user .timestamp {
            color: rgba(255, 255, 255, 0.8);
        }
        
        .message.support .timestamp {
            color: #636e72;
        }
        
        .empty-state {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            height: 100%;
            color: #636e72;
            text-align: center;
        }
        
        .empty-state i {
            font-size: 3rem;
            margin-bottom: 16px;
            opacity: 0.5;
        }
        
        .empty-state h3 {
            font-size: 1.2rem;
            font-weight: 500;
            margin-bottom: 8px;
        }
        
        .empty-state p {
            opacity: 0.7;
            margin: 0;
        }
        
        .chat-input-container {
            padding: 20px;
            background: white;
            border-top: 1px solid rgba(0, 0, 0, 0.06);
            position: relative;
        }
        
        .input-wrapper {
            position: relative;
            background: #f8f9fa;
            border-radius: 25px;
            padding: 4px;
            box-shadow: inset 0 2px 4px rgba(0, 0, 0, 0.06);
            transition: all 0.3s ease;
        }
        
        .input-wrapper:focus-within {
            background: white;
            box-shadow: 0 0 0 3px rgba(108, 92, 231, 0.1), 0 4px 12px rgba(0, 0, 0, 0.1);
        }
        
        .message-input {
            border: none;
            background: transparent;
            padding: 12px 20px;
            padding-right: 60px;
            border-radius: 25px;
            font-size: 0.95rem;
            outline: none;
            width: 100%;
            resize: none;
            min-height: 20px;
            max-height: 100px;
        }
        
        .message-input::placeholder {
            color: #636e72;
            opacity: 0.7;
        }
        
        .send-button {
            position: absolute;
            right: 6px;
            top: 50%;
            transform: translateY(-50%);
            width: 40px;
            height: 40px;
            border: none;
            border-radius: 50%;
            background: linear-gradient(135deg, #6c5ce7, #a29bfe);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .send-button:hover:not(:disabled) {
            transform: translateY(-50%) scale(1.05);
            box-shadow: 0 4px 12px rgba(108, 92, 231, 0.4);
        }
        
        .send-button:disabled {
            background: #ddd;
            cursor: not-allowed;
            transform: translateY(-50%);
        }
        
        .send-button i {
            font-size: 0.9rem;
        }
        
        .typing-indicator {
            display: none;
            align-items: center;
            gap: 8px;
            padding: 12px 18px;
            background: white;
            border-radius: 20px;
            border-bottom-left-radius: 6px;
            max-width: 80px;
            margin-bottom: 16px;
            box-shadow: 0 2px 12px rgba(0, 0, 0, 0.08);
            border: 1px solid rgba(0, 0, 0, 0.06);
        }
        
        .typing-dots {
            display: flex;
            gap: 4px;
        }
        
        .typing-dots div {
            width: 6px;
            height: 6px;
            background: #636e72;
            border-radius: 50%;
            opacity: 0.4;
            animation: typing 1.4s infinite ease-in-out;
        }
        
        .typing-dots div:nth-child(2) {
            animation-delay: 0.2s;
        }
        
        .typing-dots div:nth-child(3) {
            animation-delay: 0.4s;
        }
        
        @keyframes typing {
            0%, 60%, 100% {
                transform: scale(1);
                opacity: 0.4;
            }
            30% {
                transform: scale(1.2);
                opacity: 1;
            }
        }
        
        .manager-notice {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            height: 100vh;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            text-align: center;
            padding: 40px;
        }
        
        .manager-notice i {
            font-size: 4rem;
            margin-bottom: 24px;
            opacity: 0.8;
        }
        
        .manager-notice h2 {
            font-size: 1.8rem;
            font-weight: 600;
            margin-bottom: 12px;
        }
        
        .manager-notice p {
            opacity: 0.8;
            font-size: 1.1rem;
        }

        @media (max-width: 768px) {
            .chat-container {
                max-width: 100%;
                height: 100vh;
            }
            
            .message {
                max-width: 75%;
            }
        }
    </style>
</head>
<body class="h-100">
    <div class="chat-container">
        <div class="chat-header">
            <h2>
                <i class="fas fa-headset"></i>
                Поддержка
                <div class="status-indicator"></div>
            </h2>
        </div>
        
        <div class="chat-messages" id="chat-messages">
            <div class="empty-state">
                <i class="fas fa-comments"></i>
                <h3>Добро пожаловать!</h3>
                <p>Напишите свой первый вопрос</p>
            </div>
        </div>
        
        <div class="typing-indicator" id="typing-indicator">
            <div class="typing-dots">
                <div></div>
                <div></div>
                <div></div>
            </div>
        </div>
        
        <div class="chat-input-container">
            <div class="input-wrapper">
                <textarea 
                    id="message-input" 
                    class="message-input" 
                    placeholder="Введите сообщение..."
                    rows="1"
                ></textarea>
                <button id="send-button" class="send-button" disabled>
                    <i class="fas fa-paper-plane"></i>
                </button>
            </div>
        </div>
    </div>

    <script src="https://telegram.org/js/telegram-web-app.js"></script>
    <script>
        // Автоматическое изменение высоты textarea
        const messageInput = document.getElementById('message-input');
        messageInput.addEventListener('input', function() {
            this.style.height = 'auto';
            this.style.height = Math.min(this.scrollHeight, 100) + 'px';
            
            const sendButton = document.getElementById('send-button');
            sendButton.disabled = this.value.trim().length === 0;
        });

        // Отладка Telegram WebApp
        console.log('Telegram WebApp:', window.Telegram?.WebApp);
        console.log('initDataUnsafe:', window.Telegram?.WebApp?.initDataUnsafe);

        // Проверка, что страница открыта через Telegram WebApp
        if (!window.Telegram || !Telegram.WebApp || !Telegram.WebApp.initDataUnsafe?.user?.id) {
            document.body.innerHTML = `
                <div class="manager-notice">
                    <i class="fab fa-telegram-plane"></i>
                    <h2>Откройте в Telegram</h2>
                    <p>Данные WebApp недоступны. Проверьте консоль для деталей.</p>
                </div>
            `;
            console.error('WebApp error: Telegram.WebApp.initDataUnsafe.user.id is undefined');
        } else {
            Telegram.WebApp.ready();
            Telegram.WebApp.expand();

            const userId = Telegram.WebApp.initDataUnsafe.user.id;
            console.log('User ID:', userId);

            // Создание JSON-файла на сервере
            fetch('create_json.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: 'user_id=' + userId
            })
                .then(response => response.json())
                .then(data => {
                    if (!data.success) {
                        console.error('Failed to create JSON file:', data.message);
                    }
                })
                .catch(error => console.error('Error creating JSON file:', error));

            const chatMessages = document.getElementById('chat-messages');
            const sendButton = document.getElementById('send-button');
            const typingIndicator = document.getElementById('typing-indicator');
            
            let lastMessageCount = 0;
            let isInitialLoad = true;

            // Проверка, является ли пользователь менеджером
            fetch('check_manager.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: 'user_id=' + userId
            })
                .then(response => response.json())
                .then(data => {
                    if (data.is_manager) {
                        document.body.innerHTML = `
                            <div class="manager-notice">
                                <i class="fas fa-user-shield"></i>
                                <h2>Панель менеджера</h2>
                                <p>Вы не можете писать самому себе в поддержку</p>
                            </div>
                        `;
                    } else {
                        loadMessages();
                    }
                })
                .catch(error => {
                    console.error('Error checking manager status:', error);
                    alert('Ошибка проверки статуса пользователя');
                });

            function loadMessages() {
                // Добавляем класс загрузки только если это не первоначальная загрузка
                if (!isInitialLoad) {
                    chatMessages.classList.add('loading');
                }
                
                fetch('load_messages.php?user_id=' + userId)
                    .then(response => response.json())
                    .then(data => {
                        // Проверяем, изменилось ли количество сообщений
                        const newMessageCount = data.messages.length;
                        const hasNewMessages = newMessageCount > lastMessageCount;
                        
                        if (data.messages.length === 0) {
                            if (chatMessages.innerHTML.indexOf('empty-state') === -1) {
                                chatMessages.innerHTML = `
                                    <div class="empty-state">
                                        <i class="fas fa-comments"></i>
                                        <h3>Добро пожаловать!</h3>
                                        <p>Напишите свой первый вопрос</p>
                                    </div>
                                `;
                            }
                        } else {
                            // Если это первоначальная загрузка или есть новые сообщения
                            if (isInitialLoad || hasNewMessages) {
                                chatMessages.innerHTML = '';
                                
                                data.messages.forEach((msg, index) => {
                                    // Определяем, новое ли это сообщение
                                    const isNewMessage = hasNewMessages && index >= lastMessageCount && !isInitialLoad;
                                    addMessage(msg.text, msg.sender, msg.timestamp, false, isNewMessage);
                                });
                                
                                // Плавно показываем все сообщения
                                setTimeout(() => {
                                    const messages = chatMessages.querySelectorAll('.message');
                                    messages.forEach((msg, index) => {
                                        setTimeout(() => {
                                            msg.classList.add('loaded');
                                        }, index * 50); // Задержка для каскадного эффекта
                                    });
                                }, 50);
                            }
                        }
                        
                        lastMessageCount = newMessageCount;
                        isInitialLoad = false;
                        
                        // Убираем класс загрузки и плавно показываем контент
                        setTimeout(() => {
                            chatMessages.classList.remove('loading');
                            chatMessages.classList.add('loaded');
                            chatMessages.scrollTop = chatMessages.scrollHeight;
                        }, 100);
                    })
                    .catch(error => {
                        chatMessages.classList.remove('loading');
                        console.error('Load messages error:', error);
                    });
            }

            function addMessage(text, sender, timestamp, animate = true, isNew = false) {
                const messageDiv = document.createElement('div');
                messageDiv.className = `message ${sender}`;
                
                if (isNew) {
                    messageDiv.classList.add('new');
                }
                
                messageDiv.innerHTML = `
                    <div class="message-bubble">
                        ${text}
                    </div>
                    <div class="timestamp">${timestamp}</div>
                `;

                // Убираем empty state если он есть
                const emptyState = chatMessages.querySelector('.empty-state');
                if (emptyState) {
                    emptyState.remove();
                }

                chatMessages.appendChild(messageDiv);
                
                if (animate) {
                    // Для новых сообщений сразу показываем с анимацией
                    setTimeout(() => {
                        messageDiv.classList.add('loaded');
                        chatMessages.scrollTop = chatMessages.scrollHeight;
                    }, 50);
                }
            }

            function showTyping() {
                typingIndicator.style.display = 'flex';
                chatMessages.scrollTop = chatMessages.scrollHeight;
            }

            function hideTyping() {
                typingIndicator.style.display = 'none';
            }

            function sendMessage() {
                const text = messageInput.value.trim();
                if (text === '') return;

                // Добавляем сообщение локально
                const timestamp = new Date().toLocaleString('ru-RU', { 
                    day: '2-digit', 
                    month: '2-digit', 
                    year: 'numeric', 
                    hour: '2-digit', 
                    minute: '2-digit'
                });
                
                addMessage(text, 'user', timestamp, true, true);

                // Обновляем счетчик сообщений
                lastMessageCount++;

                // Очищаем поле ввода
                messageInput.value = '';
                messageInput.style.height = 'auto';
                sendButton.disabled = true;

                // Показываем индикатор печати
                showTyping();

                // Отправляем сообщение на сервер
                fetch('save_message.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: 'user_id=' + userId + '&text=' + encodeURIComponent(text) + '&sender=user'
                })
                    .then(response => response.json())
                    .then(data => {
                        hideTyping();
                        if (data.success) {
                            loadMessages();
                        } else {
                            alert('Ошибка отправки сообщения');
                        }
                    })
                    .catch(error => {
                        hideTyping();
                        alert('Ошибка отправки сообщения');
                        console.error('Send message error:', error);
                    });
            }

            // Автообновление каждые 5 секунд (увеличил интервал)
            setInterval(loadMessages, 5000);

            // Обработчик нажатия кнопки
            sendButton.addEventListener('click', sendMessage);
            
            // Обработчик Enter (с поддержкой Shift+Enter для новой строки)
            messageInput.addEventListener('keydown', (e) => {
                if (e.key === 'Enter' && !e.shiftKey && !sendButton.disabled) {
                    e.preventDefault();
                    sendMessage();
                }
            });
        }
    </script>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
