<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Подключение конфигурации
$config = require 'config.php';

// Подключение к базе данных
function getPDOConnection($config) {
    $dsn = "mysql:host={$config['database']['host']};dbname={$config['database']['db']};charset={$config['database']['charset']}";
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ];
    
    try {
        return new PDO($dsn, $config['database']['user'], $config['database']['pass'], $options);
    } catch (\PDOException $e) {
        throw new Exception('Ошибка подключения к базе данных');
    }
}

// Функции для работы с базой данных
function addManager($pdo, $chat_id, $name) {
    $stmt = $pdo->prepare('INSERT INTO managers (chat_id, name) VALUES (:chat_id, :name)');
    return $stmt->execute(['chat_id' => $chat_id, 'name' => $name]);
}

function addSupplier($pdo, $chat_id, $name) {
    $stmt = $pdo->prepare('INSERT INTO suppliers (chat_id, name) VALUES (:chat_id, :name)');
    return $stmt->execute(['chat_id' => $chat_id, 'name' => $name]);
}

function getManagers($pdo) {
    $stmt = $pdo->query('SELECT id, chat_id, name FROM managers ORDER BY name');
    return $stmt->fetchAll();
}

function getSuppliers($pdo) {
    $stmt = $pdo->query('SELECT id, chat_id, name FROM suppliers ORDER BY name');
    return $stmt->fetchAll();
}

function deleteManager($pdo, $id) {
    $stmt = $pdo->prepare('DELETE FROM managers WHERE id = :id');
    return $stmt->execute(['id' => $id]);
}

function deleteSupplier($pdo, $id) {
    $stmt = $pdo->prepare('DELETE FROM suppliers WHERE id = :id');
    return $stmt->execute(['id' => $id]);
}

// Получение данных запроса
$input = json_decode(file_get_contents('php://input'), true);
$action = $input['action'] ?? '';

try {
    $pdo = getPDOConnection($config);
    
    switch ($action) {
        case 'checkAccess':
            $user = $input['user'] ?? null;
            
            if (!$user || !isset($user->id)) {
                throw new Exception('Нет данных пользователя');
            }
            
            // Проверка прав администратора из config.php
            if (in_array((int)$user->id, $config['admin_ids'])) {
                echo json_encode(['success' => true, 'message' => 'Доступ разрешен']);
            } else {
                throw new Exception('Доступ запрещен');
            }
            break;
            
        case 'addManager':
            $chat_id = $input['chat_id'] ?? '';
            $name = $input['name'] ?? '';
            
            if (empty($chat_id) || empty($name)) {
                throw new Exception('Заполните все поля');
            }
            
            if (!is_numeric($chat_id)) {
                throw new Exception('Chat ID должен быть числом');
            }
            
            // Проверка на дубликаты
            $stmt = $pdo->prepare('SELECT id FROM managers WHERE chat_id = :chat_id');
            $stmt->execute(['chat_id' => $chat_id]);
            if ($stmt->fetch()) {
                throw new Exception('Менеджер с таким Chat ID уже существует');
            }
            
            addManager($pdo, $chat_id, $name);
            echo json_encode(['success' => true, 'message' => 'Менеджер добавлен']);
            break;
            
        case 'addSupplier':
            $chat_id = $input['chat_id'] ?? '';
            $name = $input['name'] ?? '';
            
            if (empty($chat_id) || empty($name)) {
                throw new Exception('Заполните все поля');
            }
            
            if (!is_numeric($chat_id)) {
                throw new Exception('Chat ID должен быть числом');
            }
            
            // Проверка на дубликаты
            $stmt = $pdo->prepare('SELECT id FROM suppliers WHERE chat_id = :chat_id');
            $stmt->execute(['chat_id' => $chat_id]);
            if ($stmt->fetch()) {
                throw new Exception('Поставщик с таким Chat ID уже существует');
            }
            
            addSupplier($pdo, $chat_id, $name);
            echo json_encode(['success' => true, 'message' => 'Поставщик добавлен']);
            break;
            
        case 'getManagers':
            $managers = getManagers($pdo);
            echo json_encode(['success' => true, 'data' => $managers]);
            break;
            
        case 'getSuppliers':
            $suppliers = getSuppliers($pdo);
            echo json_encode(['success' => true, 'data' => $suppliers]);
            break;
            
        case 'deleteManager':
            $id = $input['id'] ?? '';
            
            if (empty($id) || !is_numeric($id)) {
                throw new Exception('Некорректный ID');
            }
            
            deleteManager($pdo, $id);
            echo json_encode(['success' => true, 'message' => 'Менеджер удален']);
            break;
            
        case 'deleteSupplier':
            $id = $input['id'] ?? '';
            
            if (empty($id) || !is_numeric($id)) {
                throw new Exception('Некорректный ID');
            }
            
            deleteSupplier($pdo, $id);
            echo json_encode(['success' => true, 'message' => 'Поставщик удален']);
            break;
            
        default:
            throw new Exception('Неизвестное действие');
    }
    
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}