<?php
// Папка с JSON-файлами переписок
$supportDir = 'support/';
$chatFiles = array_diff(scandir($supportDir), ['.', '..']);
$chats = [];

foreach ($chatFiles as $file) {
    if (pathinfo($file, PATHINFO_EXTENSION) === 'json') {
        $userId = pathinfo($file, PATHINFO_FILENAME);
        $data = json_decode(file_get_contents($supportDir . $file), true);
        $lastMessage = !empty($data['messages']) ? end($data['messages']) : ['timestamp' => '01.01.1970 00:00:00', 'text' => 'Нет сообщений', 'sender' => 'system'];
        $chats[] = [
            'user_id' => $userId,
            'last_message_time' => $lastMessage['timestamp'],
            'last_message_text' => $lastMessage['text'],
            'last_message_sender' => $lastMessage['sender']
        ];
    }
}

// Сортировка чатов по времени последнего сообщения (новые вверху)
usort($chats, function ($a, $b) {
    return strtotime($b['last_message_time']) - strtotime($a['last_message_time']);
});

/**
 * Функция отправки уведомления пользователю через Telegram Bot
 */
function sendTelegramNotification($userId, $message = null) {
    try {
        $config = require 'config.php';
        $botToken = $config['telegram']['token'];
        
        if (empty($botToken)) {
            error_log("Telegram bot token not found in config");
            return false;
        }
        
        $text = "📨 *Вам пришло сообщение от поддержки*";
        
        if ($message && strlen($message) > 0) {
            // Обрезаем сообщение если слишком длинное и экранируем markdown
            $preview = strlen($message) > 50 ? substr($message, 0, 50) . "..." : $message;
            $preview = str_replace(['_', '*', '[', ']', '(', ')', '~', '`', '>', '#', '+', '-', '=', '|', '{', '}', '.', '!'], 
                                 ['\\_', '\\*', '\\[', '\\]', '\\(', '\\)', '\\~', '\\`', '\\>', '\\#', '\\+', '\\-', '\\=', '\\|', '\\{', '\\}', '\\.', '\\!'], 
                                 $preview);
            $text .= "\n\n💬 " . $preview;
        }
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    [
                        'text' => '💬 Открыть чат',
                        'url' => 'https://forzabot.pro/client_webapp.php'
                    ]
                ]
            ]
        ];
        
        $data = [
            'chat_id' => $userId,
            'text' => $text,
            'parse_mode' => 'Markdown',
            'reply_markup' => json_encode($keyboard)
        ];
        
        $url = "https://api.telegram.org/bot{$botToken}/sendMessage";
        
        $options = [
            'http' => [
                'header' => "Content-type: application/x-www-form-urlencoded\r\n",
                'method' => 'POST',
                'content' => http_build_query($data),
                'timeout' => 10
            ]
        ];
        
        $context = stream_context_create($options);
        $result = @file_get_contents($url, false, $context);
        
        if ($result === false) {
            error_log("Failed to send Telegram notification to user {$userId}");
            return false;
        }
        
        $response = json_decode($result, true);
        
        if (!$response['ok']) {
            error_log("Telegram API error for user {$userId}: " . $response['description']);
            return false;
        }
        
        error_log("Telegram notification sent successfully to user {$userId}");
        return true;
        
    } catch (Exception $e) {
        error_log("Exception sending Telegram notification: " . $e->getMessage());
        return false;
    }
}
?>

<!DOCTYPE html>
<html lang="ru" class="h-100">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Панель менеджера</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        * {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        }
        
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            height: 100vh;
            margin: 0;
            overflow: hidden;
        }
        
        .chat-container {
            height: 100vh;
            max-width: 500px;
            margin: 0 auto;
            display: flex;
            flex-direction: column;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            box-shadow: 0 25px 50px rgba(0, 0, 0, 0.15);
        }
        
        .chat-header {
            background: linear-gradient(135deg, #6c5ce7, #a29bfe);
            padding: 20px;
            border-radius: 0;
            position: relative;
            overflow: hidden;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }
        
        .chat-header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grid" width="10" height="10" patternUnits="userSpaceOnUse"><path d="M 10 0 L 0 0 0 10" fill="none" stroke="rgba(255,255,255,0.1)" stroke-width="0.5"/></pattern></defs><rect width="100" height="100" fill="url(%23grid)"/></svg>');
            opacity: 0.3;
        }
        
        .chat-header h2 {
            margin: 0;
            color: white;
            font-weight: 600;
            font-size: 1.5rem;
            position: relative;
            z-index: 2;
            display: flex;
            align-items: center;
            gap: 12px;
        }
        
        .back-button {
            background: rgba(255, 255, 255, 0.2);
            border: none;
            color: white;
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s ease;
            backdrop-filter: blur(10px);
            position: relative;
            z-index: 2;
        }
        
        .back-button:hover {
            background: rgba(255, 255, 255, 0.3);
            transform: scale(1.05);
        }
        
        .status-indicator {
            width: 12px;
            height: 12px;
            background: #00b894;
            border-radius: 50%;
            box-shadow: 0 0 10px rgba(0, 184, 148, 0.6);
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.1); }
            100% { transform: scale(1); }
        }
        
        .chat-messages {
            flex: 1;
            padding: 20px;
            overflow-y: auto;
            background: linear-gradient(to bottom, rgba(255, 255, 255, 0.1), rgba(247, 250, 255, 0.8));
            position: relative;
        }
        
        .chat-messages::-webkit-scrollbar {
            width: 6px;
        }
        
        .chat-messages::-webkit-scrollbar-track {
            background: rgba(0, 0, 0, 0.05);
            border-radius: 10px;
        }
        
        .chat-messages::-webkit-scrollbar-thumb {
            background: linear-gradient(135deg, #6c5ce7, #a29bfe);
            border-radius: 10px;
        }
        
        .message {
            max-width: 280px;
            margin-bottom: 16px;
            position: relative;
            opacity: 0;
            transform: translateY(10px);
            transition: all 0.3s ease-out;
        }
        
        .message.loaded {
            opacity: 1;
            transform: translateY(0);
        }
        
        .message.new {
            animation: slideIn 0.4s ease-out forwards;
        }
        
        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(20px) scale(0.95);
            }
            to {
                opacity: 1;
                transform: translateY(0) scale(1);
            }
        }
        
        .chat-messages.loading {
            opacity: 0.7;
            pointer-events: none;
        }
        
        .chat-messages.loaded {
            opacity: 1;
            transition: opacity 0.3s ease;
        }
        
        .message.user {
            margin-right: auto;
            margin-left: 0;
        }
        
        .message.support {
            margin-left: auto;
            margin-right: 0;
        }
        
        .message-bubble {
            padding: 14px 18px;
            border-radius: 20px;
            position: relative;
            word-wrap: break-word;
            box-shadow: 0 2px 12px rgba(0, 0, 0, 0.08);
        }
        
        .message.user .message-bubble {
            background: linear-gradient(135deg, #6c5ce7, #a29bfe);
            color: white;
            border-bottom-left-radius: 6px;
        }
        
        .message.user .message-bubble::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: -8px;
            width: 0;
            height: 0;
            border: 8px solid transparent;
            border-right-color: #a29bfe;
            border-bottom: 0;
            transform: rotate(-45deg);
        }
        
        .message.support .message-bubble {
            background: linear-gradient(135deg, #00b894, #00cec9);
            color: white;
            border-bottom-right-radius: 6px;
        }
        
        .message.support .message-bubble::after {
            content: '';
            position: absolute;
            bottom: 0;
            right: -8px;
            width: 0;
            height: 0;
            border: 8px solid transparent;
            border-left-color: #00cec9;
            border-bottom: 0;
            transform: rotate(45deg);
        }
        
        .timestamp {
            font-size: 0.75rem;
            opacity: 0.6;
            margin-top: 4px;
            text-align: center;
        }
        
        .message.support .timestamp {
            color: rgba(255, 255, 255, 0.8);
        }
        
        .message.user .timestamp {
            color: #636e72;
        }
        
        .empty-state {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            height: 100%;
            color: #636e72;
            text-align: center;
        }
        
        .empty-state i {
            font-size: 3rem;
            margin-bottom: 16px;
            opacity: 0.5;
        }
        
        .empty-state h3 {
            font-size: 1.2rem;
            font-weight: 500;
            margin-bottom: 8px;
        }
        
        .empty-state p {
            opacity: 0.7;
            margin: 0;
        }
        
        .chat-input-container {
            padding: 20px;
            background: white;
            border-top: 1px solid rgba(0, 0, 0, 0.06);
            position: relative;
        }
        
        .input-wrapper {
            position: relative;
            background: #f8f9fa;
            border-radius: 25px;
            padding: 4px;
            box-shadow: inset 0 2px 4px rgba(0, 0, 0, 0.06);
            transition: all 0.3s ease;
        }
        
        .input-wrapper:focus-within {
            background: white;
            box-shadow: 0 0 0 3px rgba(108, 92, 231, 0.1), 0 4px 12px rgba(0, 0, 0, 0.1);
        }
        
        .message-input {
            border: none;
            background: transparent;
            padding: 12px 20px;
            padding-right: 60px;
            border-radius: 25px;
            font-size: 0.95rem;
            outline: none;
            width: 100%;
            resize: none;
            min-height: 20px;
            max-height: 100px;
        }
        
        .message-input::placeholder {
            color: #636e72;
            opacity: 0.7;
        }
        
        .send-button {
            position: absolute;
            right: 6px;
            top: 50%;
            transform: translateY(-50%);
            width: 40px;
            height: 40px;
            border: none;
            border-radius: 50%;
            background: linear-gradient(135deg, #6c5ce7, #a29bfe);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .send-button:hover:not(:disabled) {
            transform: translateY(-50%) scale(1.05);
            box-shadow: 0 4px 12px rgba(108, 92, 231, 0.4);
        }
        
        .send-button:disabled {
            background: #ddd;
            cursor: not-allowed;
            transform: translateY(-50%);
        }
        
        .send-button i {
            font-size: 0.9rem;
        }
        
        .typing-indicator {
            display: none;
            align-items: center;
            gap: 8px;
            padding: 12px 18px;
            background: white;
            border-radius: 20px;
            border-bottom-left-radius: 6px;
            max-width: 80px;
            margin-bottom: 16px;
            box-shadow: 0 2px 12px rgba(0, 0, 0, 0.08);
            border: 1px solid rgba(0, 0, 0, 0.06);
        }
        
        .typing-dots {
            display: flex;
            gap: 4px;
        }
        
        .typing-dots div {
            width: 6px;
            height: 6px;
            background: #636e72;
            border-radius: 50%;
            opacity: 0.4;
            animation: typing 1.4s infinite ease-in-out;
        }
        
        .typing-dots div:nth-child(2) {
            animation-delay: 0.2s;
        }
        
        .typing-dots div:nth-child(3) {
            animation-delay: 0.4s;
        }
        
        @keyframes typing {
            0%, 60%, 100% {
                transform: scale(1);
                opacity: 0.4;
            }
            30% {
                transform: scale(1.2);
                opacity: 1;
            }
        }

        /* Список чатов */
        .chat-list-container {
            flex: 1;
            overflow-y: auto;
            background: linear-gradient(to bottom, rgba(255, 255, 255, 0.1), rgba(247, 250, 255, 0.8));
        }
        
        .chat-list-container::-webkit-scrollbar {
            width: 6px;
        }
        
        .chat-list-container::-webkit-scrollbar-track {
            background: rgba(0, 0, 0, 0.05);
        }
        
        .chat-list-container::-webkit-scrollbar-thumb {
            background: linear-gradient(135deg, #6c5ce7, #a29bfe);
            border-radius: 10px;
        }
        
        .chat-item {
            display: flex;
            align-items: center;
            padding: 16px 20px;
            border: none;
            background: transparent;
            width: 100%;
            text-align: left;
            cursor: pointer;
            transition: all 0.3s ease;
            border-bottom: 1px solid rgba(0, 0, 0, 0.06);
        }
        
        .chat-item:hover {
            background: rgba(108, 92, 231, 0.08);
            transform: translateX(4px);
        }
        
        .avatar {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background: linear-gradient(135deg, #6c5ce7, #a29bfe);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
            font-size: 1.1rem;
            margin-right: 15px;
            box-shadow: 0 4px 12px rgba(108, 92, 231, 0.3);
            position: relative;
        }
        
        .avatar::after {
            content: '';
            position: absolute;
            bottom: 2px;
            right: 2px;
            width: 14px;
            height: 14px;
            background: #00b894;
            border-radius: 50%;
            border: 3px solid white;
            box-shadow: 0 0 8px rgba(0, 184, 148, 0.6);
        }
        
        .chat-info {
            flex: 1;
            min-width: 0;
        }
        
        .chat-header-info {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 4px;
        }
        
        .client-name {
            font-weight: 600;
            color: #2d3436;
            font-size: 1rem;
        }
        
        .time {
            font-size: 0.8rem;
            color: #636e72;
            opacity: 0.8;
        }
        
        .last-message {
            color: #636e72;
            font-size: 0.85rem;
            margin: 0;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            line-height: 1.3;
        }
        
        .last-message.user-message {
            color: #6c5ce7;
            font-weight: 500;
        }

        @media (max-width: 768px) {
            .chat-container {
                max-width: 100%;
                height: 100vh;
            }
            
            .message {
                max-width: 75%;
            }
            
            .chat-header h2 {
                font-size: 1.3rem;
            }
            
            .back-button {
                width: 36px;
                height: 36px;
            }
        }
    </style>
</head>
<body class="h-100">
    <div class="chat-container">
        <!-- Список чатов -->
        <div id="chat-list-view" style="display: flex; flex-direction: column; height: 100vh;">
            <div class="chat-header">
                <h2>
                    <i class="fas fa-headset"></i>
                    Чаты поддержки
                    <div class="status-indicator"></div>
                </h2>
            </div>
            
            <div class="chat-list-container">
                <?php if (!empty($chats)): ?>
                    <?php foreach ($chats as $chat): ?>
                        <button class="chat-item" onclick="openChat('<?php echo htmlspecialchars($chat['user_id']); ?>')">
                            <div class="avatar">
                                <i class="fas fa-user"></i>
                            </div>
                            <div class="chat-info">
                                <div class="chat-header-info">
                                    <span class="client-name">Клиент <?php echo substr($chat['user_id'], -4); ?></span>
                                    <span class="time"><?php echo date('H:i', strtotime($chat['last_message_time'])); ?></span>
                                </div>
                                <p class="last-message <?php echo $chat['last_message_sender'] === 'user' ? 'user-message' : ''; ?>">
                                    <?php 
                                    $message = htmlspecialchars($chat['last_message_text']);
                                    echo strlen($message) > 45 ? substr($message, 0, 45) . '...' : $message;
                                    ?>
                                </p>
                            </div>
                        </button>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div class="empty-state">
                        <i class="fas fa-inbox"></i>
                        <h3>Нет активных чатов</h3>
                        <p>Здесь появятся обращения клиентов</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Чат с клиентом -->
        <div id="chat-view" style="display: none; flex-direction: column; height: 100vh;">
            <div class="chat-header">
                <button class="back-button" onclick="backToList()">
                    <i class="fas fa-arrow-left"></i>
                </button>
                <h2 id="chat-title">
                    <i class="fas fa-user-circle"></i>
                    Клиент
                    <div class="status-indicator"></div>
                </h2>
            </div>
            
            <div class="chat-messages" id="chat-messages">
                <div class="empty-state">
                    <i class="fas fa-comments"></i>
                    <h3>Начните переписку</h3>
                    <p>Напишите первое сообщение клиенту</p>
                </div>
            </div>
            
            <div class="typing-indicator" id="typing-indicator">
                <div class="typing-dots">
                    <div></div>
                    <div></div>
                    <div></div>
                </div>
            </div>
            
            <div class="chat-input-container">
                <div class="input-wrapper">
                    <textarea 
                        id="message-input" 
                        class="message-input" 
                        placeholder="Введите ответ клиенту..."
                        rows="1"
                    ></textarea>
                    <button id="send-button" class="send-button" disabled>
                        <i class="fas fa-paper-plane"></i>
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://telegram.org/js/telegram-web-app.js"></script>
    <script>
        // Автоматическое изменение высоты textarea
        const messageInput = document.getElementById('message-input');
        const sendButton = document.getElementById('send-button');
        
        messageInput.addEventListener('input', function() {
            this.style.height = 'auto';
            this.style.height = Math.min(this.scrollHeight, 100) + 'px';
            sendButton.disabled = this.value.trim().length === 0;
        });

        // Проверка WebApp
        if (!window.Telegram || !Telegram.WebApp || !Telegram.WebApp.initDataUnsafe?.user?.id) {
            document.body.innerHTML = `
                <div class="chat-container">
                    <div class="chat-header">
                        <h2><i class="fab fa-telegram-plane"></i> Откройте в Telegram</h2>
                    </div>
                    <div class="empty-state" style="flex: 1; display: flex;">
                        <i class="fab fa-telegram-plane"></i>
                        <h3>WebApp недоступен</h3>
                        <p>Откройте через Telegram</p>
                    </div>
                </div>
            `;
            console.error('WebApp error: Telegram.WebApp.initDataUnsafe.user.id is undefined');
        } else {
            Telegram.WebApp.ready();
            Telegram.WebApp.expand();

            const chatListView = document.getElementById('chat-list-view');
            const chatView = document.getElementById('chat-view');
            const chatMessages = document.getElementById('chat-messages');
            const chatTitle = document.getElementById('chat-title');
            const typingIndicator = document.getElementById('typing-indicator');
            
            let currentUserId = null;
            let lastMessageCount = 0;
            let isInitialLoad = true;

            function openChat(userId) {
                currentUserId = userId;
                chatListView.style.display = 'none';
                chatView.style.display = 'flex';
                chatView.style.flexDirection = 'column';
                chatTitle.innerHTML = `<i class="fas fa-user-circle"></i> Клиент ${userId.substr(-4)} <div class="status-indicator"></div>`;
                
                isInitialLoad = true;
                lastMessageCount = 0;
                loadMessages();
            }

            function backToList() {
                chatListView.style.display = 'flex';
                chatListView.style.flexDirection = 'column';
                chatView.style.display = 'none';
                currentUserId = null;
            }

            function loadMessages() {
                if (!currentUserId) return;

                if (!isInitialLoad) {
                    chatMessages.classList.add('loading');
                }
                
                fetch('load_messages.php?user_id=' + currentUserId)
                    .then(response => response.json())
                    .then(data => {
                        const newMessageCount = data.messages.length;
                        const hasNewMessages = newMessageCount > lastMessageCount;
                        
                        if (data.messages.length === 0) {
                            if (chatMessages.innerHTML.indexOf('empty-state') === -1) {
                                chatMessages.innerHTML = `
                                    <div class="empty-state">
                                        <i class="fas fa-comments"></i>
                                        <h3>Начните переписку</h3>
                                        <p>Напишите первое сообщение клиенту</p>
                                    </div>
                                `;
                            }
                        } else {
                            if (isInitialLoad || hasNewMessages) {
                                chatMessages.innerHTML = '';
                                
                                data.messages.forEach((msg, index) => {
                                    const isNewMessage = hasNewMessages && index >= lastMessageCount && !isInitialLoad;
                                    addMessage(msg.text, msg.sender, msg.timestamp, false, isNewMessage);
                                });
                                
                                setTimeout(() => {
                                    const messages = chatMessages.querySelectorAll('.message');
                                    messages.forEach((msg, index) => {
                                        setTimeout(() => {
                                            msg.classList.add('loaded');
                                        }, index * 50);
                                    });
                                }, 50);
                            }
                        }
                        
                        lastMessageCount = newMessageCount;
                        isInitialLoad = false;
                        
                        setTimeout(() => {
                            chatMessages.classList.remove('loading');
                            chatMessages.classList.add('loaded');
                            chatMessages.scrollTop = chatMessages.scrollHeight;
                        }, 100);
                    })
                    .catch(error => {
                        chatMessages.classList.remove('loading');
                        console.error('Load messages error:', error);
                    });
            }

            function addMessage(text, sender, timestamp, animate = true, isNew = false) {
                const messageDiv = document.createElement('div');
                messageDiv.className = `message ${sender}`;
                
                if (isNew) {
                    messageDiv.classList.add('new');
                }
                
                messageDiv.innerHTML = `
                    <div class="message-bubble">
                        ${text}
                    </div>
                    <div class="timestamp">${timestamp}</div>
                `;

                const emptyState = chatMessages.querySelector('.empty-state');
                if (emptyState) {
                    emptyState.remove();
                }

                chatMessages.appendChild(messageDiv);
                
                if (animate) {
                    setTimeout(() => {
                        messageDiv.classList.add('loaded');
                        chatMessages.scrollTop = chatMessages.scrollHeight;
                    }, 50);
                }
            }

            function showTyping() {
                typingIndicator.style.display = 'flex';
                chatMessages.scrollTop = chatMessages.scrollHeight;
            }

            function hideTyping() {
                typingIndicator.style.display = 'none';
            }

            function sendMessage() {
                const text = messageInput.value.trim();
                if (text === '' || !currentUserId) return;

                const timestamp = new Date().toLocaleString('ru-RU', { 
                    day: '2-digit', 
                    month: '2-digit', 
                    year: 'numeric', 
                    hour: '2-digit', 
                    minute: '2-digit'
                });
                
                addMessage(text, 'support', timestamp, true, true);
                lastMessageCount++;

                messageInput.value = '';
                messageInput.style.height = 'auto';
                sendButton.disabled = true;

                showTyping();

                fetch('save_message.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: 'user_id=' + currentUserId + '&text=' + encodeURIComponent(text) + '&sender=support'
                })
                    .then(response => response.json())
                    .then(data => {
                        hideTyping();
                        if (data.success) {
                            loadMessages();
                        } else {
                            alert('Ошибка отправки сообщения');
                        }
                    })
                    .catch(error => {
                        hideTyping();
                        alert('Ошибка отправки сообщения');
                        console.error('Send message error:', error);
                    });
            }

            // Автообновление каждые 5 секунд
            setInterval(() => {
                if (currentUserId) {
                    loadMessages();
                }
            }, 5000);

            sendButton.addEventListener('click', sendMessage);
            
            messageInput.addEventListener('keydown', (e) => {
                if (e.key === 'Enter' && !e.shiftKey && !sendButton.disabled) {
                    e.preventDefault();
                    sendMessage();
                }
            });
        }
    </script>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>